unit WinIOCtl;

interface

uses
  Windows, SysUtils;

function EjectMediaInDrive(nDrive: Byte): Boolean;

implementation

// for Windows NT
function CTL_CODE(DeviceType,Func,Method,Access: DWord): DWord;
begin
  Result := (DeviceType shl 16) or (Access shl 14) or (Func shl 2) or (Method);
end;

const
  FILE_DEVICE_DISK = $00000007;
  METHOD_BUFFERED  = 0;
  FILE_READ_ACCESS = $0001; // file & pipe
  IOCTL_DISK_BASE  = FILE_DEVICE_DISK;

function IOCTL_DISK_EJECT_MEDIA: DWord;
begin
  Result := CTL_CODE(IOCTL_DISK_BASE,$0202,METHOD_BUFFERED,FILE_READ_ACCESS);
end;

function IOCTL_DISK_LOAD_MEDIA: DWord;
begin
  Result := CTL_CODE(IOCTL_DISK_BASE,$0203,METHOD_BUFFERED,FILE_READ_ACCESS);
end;

function EjectMediaInDriveNT(nDrive: Byte): Boolean;
var
  hDev: THandle;
  dwRet: DWORD;
  DeviceDiskFileName: string;
begin
  Result := False;
  DeviceDiskFileName := '\\.\'+Chr(Ord('@')+nDrive)+':';
  hDev := CreateFile(PChar(DeviceDiskFileName),GENERIC_READ,FILE_SHARE_READ or FILE_SHARE_WRITE,nil,OPEN_EXISTING,0,0);
  if INVALID_HANDLE_VALUE <> hDev then begin
    if DeviceIoControl(hDev,IOCTL_DISK_EJECT_MEDIA,nil,0,nil,0,dwRet,nil) then begin
      Result := True;
    end;
    CloseHandle(hDev)
  end;
end;

// for Windows 95
const
  VWIN32_DIOC_DOS_IOCTL = 1;

type
  PDEVIOCTL_REGISTERS = ^TDEVIOCTL_REGISTERS;
  TDEVIOCTL_REGISTERS = packed record
    reg_EBX: DWord;
    reg_EDX: DWord;
    reg_ECX: DWord;
    reg_EAX: DWord;
    reg_EDI: DWord;
    reg_ESI: DWord;
    reg_Flags: DWord;
  end;

function DoIOCTL(var Reg: TDEVIOCTL_REGISTERS): Boolean;
var
  hDevice: THandle;
  fResult: Boolean;
  CB: DWord;
begin
  Result := False;
  Reg.reg_Flags := $8000; // assume error (carry flag set)
  hDevice := CreateFile(
    '\\.\vwin32',
    GENERIC_READ,
    FILE_SHARE_READ or FILE_SHARE_WRITE,
    PSecurityAttributes(nil),
    OPEN_EXISTING,
    FILE_ATTRIBUTE_NORMAL,
    THandle(nil)
  );
  if hDevice <> THandle(INVALID_HANDLE_VALUE) then begin
    fResult := DeviceIoControl(
      hDevice,
      VWIN32_DIOC_DOS_IOCTL,
      @Reg,SizeOf(Reg),
      @Reg,SizeOf(Reg),
      CB,
      nil
    );
    if fResult then begin
      CloseHandle(hDevice);
      Result := True;
    end;
  end;
end;

{ Eject media in drive(GENERIC BLOCK DEVICE REQUEST) }
{ nDrive: drive number (00h=default,01h=A:,etc) }
function EjectMediaInDrive95(nDrive: Byte): Boolean;
var
  Reg: TDEVIOCTL_REGISTERS;
begin
  Result := False;
(*
 * From Ralf Brown's Interrupt List Release 53
 *
 * INT 21 440D - DOS 3.2+ - IOCTL - GENERIC BLOCK DEVICE REQUEST
 *   AX = 440Dh
 *   BL = drive number (00h=default,01h=A:,etc)
 *   CH = category code (08h:disk drive, 48h:FAT32 disk drive, ...)
 *   CL = minor code (function) (..., 49h:(Enh. Disk Drive Spec) eject media in drive (see INT 13/AH=49h), ...)
 *   DS:DX -> (DOS) parameter block
 *   SI:DI -> (OS/2 comp box) parameter block
 * Return: CF set on error
 *     AX = error code (01h,02h,etc.)
 *     CF clear if successful
 *	 DS:DX -> data block if CL=60h or CL=61h
 *)
  with Reg do begin
    reg_EAX := $440D;       // IOCTL for block devices
    reg_EBX := nDrive;      // zero-based drive ID
    reg_ECX := $0849;       // eject media in drive ($4849 for FAT32 disk drive?)
    reg_EDX := 0;           // no parameter block required
  end;
  if DoIOCTL(Reg) then begin
    // error if carry flag set
    if 0 = (Reg.reg_Flags and $8000) then Result := True;
    // Result := ($FFFF & Reg.reg_EAX);
  end;
end;

// Windows NT and Windows 95
function EjectMediaInDrive(nDrive: Byte): Boolean;
var
  nErrMode: UINT;
begin
  Result := False;
  nErrMode := SetErrorMode(SEM_FAILCRITICALERRORS);
  try
    case SysUtils.Win32Platform of
      VER_PLATFORM_WIN32_WINDOWS: Result := EjectMediaInDrive95(nDrive);
      VER_PLATFORM_WIN32_NT:      Result := EjectMediaInDriveNT(nDrive);
    end;
  finally
    SetErrorMode(nErrMode);
  end;
end;

end.
